<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Contacto;
use App\Models\ContactoEmpresa;
use App\Models\Experiencia;
use App\Models\Place;
use App\Models\UsedIp;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Stevebauman\Location\Facades\Location;
use Carbon\Carbon;


class HomeController extends Controller
{
    public static function home($modal = false)
    {

        return view('welcome',[
            'modal'=>$modal,
            'lugares'=>Place::withCount('experiencias')->where('status', 1)->orderBy('experiencias_count', 'desc')->take(8)->get(),
            'categorias'=>Category::all(),
            'experiencias'=>Experiencia::where('main_page', 1)
                ->orderBy('id', 'desc')
                ->take(8)
                ->get(),
        ]);
    }

    public function categorias($id)
    {
        return view('experiencias.categorias', [
        ]);
    }

    public function formulario_contacto()
    {
        return view('formularios.contacto');
    }

    public function procesar_formulario_contacto(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email',
            'phone' => 'required',
        ]);

        if (!$validator->passes()) {
            return back()->withErrors($validator->errors());
        }

        $clientIp = $request->ip();
        $ip_first = UsedIp::where('ip', $clientIp)
            ->where('created_at', '>', Carbon::now()->subMinutes(60)->toDateTimeString())
            ->first();

        if ($ip_first) {
            return back()->withErrors(["Error"=>"Ya ha enviado un contacto en la última hora. Espere unos minutos para volver a contactarse."]);
        }

        $contacto = new Contacto();
        $contacto->name=$request->name;
        $contacto->phone=$request->phone;
        $contacto->email=$request->email;
        $contacto->message=$request->message;
        $contacto->save();

        UsedIp::create([
            'ip'=>$clientIp,
            'times_used'=>1,
        ]);

        //enviar mail
        try {
            ContactMailController::send_mail($contacto->mensaje,$contacto->name,$contacto->phone,$contacto->email);
            return view('formularios.contacto',[
                'resultado'=>"Su mensaje ha sido enviado satisfactoriamente. <br />
                    La administración de Yeah! se pondrá en contacto a la brevedad."
            ]);
        } catch (\Throwable $th) {
            return view('formularios.contacto',[
                'resultado'=>"Su mensaje no ha sido enviado . <br />
                        Error.".$th
                ]);
        }
    }

    public function formulario_empresas()
    {
        return view('formularios.contacto-empresas');
    }

    public function procesar_formulario_empresas(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email',
            'phone' => 'required',
        ]);

        if (!$validator->passes()) {
            return back()->withErrors($validator->errors());
        }

        $clientIp = $request->ip();
        $ip_first = UsedIp::where('ip', $clientIp)
            ->where('created_at', '>', Carbon::now()->subMinutes(60)->toDateTimeString())
            ->first();

        if ($ip_first) {
            return back()->withErrors(["Error"=>"Ya ha enviado un contacto en la última hora. Espere unos minutos para volver a contactarse."]);
        }

        $contacto = new ContactoEmpresa();
        $contacto->name=$request->name;
        $contacto->phone=$request->phone;
        $contacto->email=$request->email;
        $contacto->company=$request->company;
        $contacto->cant_empleados=$request->cant_empleados;
        $contacto->message=$request->message;
        $contacto->save();

        UsedIp::create([
            'ip'=>$clientIp,
            'times_used'=>1,
        ]);

        //enviar mail
        try {
            ContactMailController::send_mail($contacto->mensaje,$contacto->name,$contacto->phone,$contacto->email,$contacto->company,$contacto->cant_empleados);
            return view('formularios.contacto-empresas',[
                'resultado'=>"Gracias por su interés. <br /> Su mensaje ha sido enviado satisfactoriamente.
                    <br /> La administración de Yeah! se pondrá en contacto a la brevedad."
            ]);
        } catch (\Throwable $th) {
            return view('formularios.contacto',[
                'resultado'=>"Su mensaje no ha sido enviado. <br /> Intentelo nuevamente más tarde."
            ]);
        }
    }

    public function categoria($id)
    {
        $categoria = Category::find($id);

        if(!$categoria){
            return view('errors.404');
        }
        $experiencias = Experiencia::with('categories')
            ->where('valid', 1)
            ->whereHas('categories', function ($q) use ($categoria) {
            $q->where('categorie_id', $categoria->id);
        })->paginate(5);

        $coordenadas = $experiencias->pluck('coordenates');

        if(!$coordenadas->isEmpty()){
            $coord_iniciales = explode(',', $coordenadas[0]);

            return view('experiencias.categoria', [
                'experiencias'=>$experiencias,
                'categoria'=>$categoria,
                'coordenadas'=>$coordenadas,
                'coord_iniciales'=>$coord_iniciales,
            ]);
        }
        else{
            return view('experiencias.categoria-sin-experiencias', ['categoria'=>$categoria]);
        }
    }

    public function lugar($id)
    {
        $lugar = Place::find($id);
        if(!$lugar){
            return view('errors.404');
        }
        $experiencias = Experiencia::where('place_id', $lugar->id)->where('valid', 1)->paginate(8);
        $coordenadas = $experiencias->pluck('coordenates');
        if(!$coordenadas->isEmpty()){
            $coord_iniciales = explode(', ', $coordenadas[0]);
            return view('experiencias.lugar', [
                'experiencias'=>$experiencias,
                'lugar'=>$lugar,
                'coordenadas'=>$coordenadas,
                'coord_iniciales'=>$coord_iniciales,
            ]);
        }
        else{
            return view('experiencias.lugar-sin-experiencias');
        }
    }

    public function notFound()
    {
        return view('not-found');
    }
}
