<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Handles registering, displaying, and dismissing notices in the admin.
 *
 * @since 3.44.3
 */
class WPF_ISO_Regions {

	/**
	 * Mapping of country names to ISO 3166-1 alpha-3 codes.
	 *
	 * @var array
	 */
	private $name_to_alpha3 = array(
		'Afghanistan'                                  => 'AFG',
		'Aland Islands'                                => 'ALA',
		'Albania'                                      => 'ALB',
		'Algeria'                                      => 'DZA',
		'American Samoa'                               => 'ASM',
		'Andorra'                                      => 'AND',
		'Angola'                                       => 'AGO',
		'Anguilla'                                     => 'AIA',
		'Antarctica'                                   => 'ATA',
		'Antigua and Barbuda'                          => 'ATG',
		'Argentina'                                    => 'ARG',
		'Armenia'                                      => 'ARM',
		'Aruba'                                        => 'ABW',
		'Australia'                                    => 'AUS',
		'Austria'                                      => 'AUT',
		'Azerbaijan'                                   => 'AZE',
		'Bahamas'                                      => 'BHS',
		'Bahrain'                                      => 'BHR',
		'Bangladesh'                                   => 'BGD',
		'Barbados'                                     => 'BRB',
		'Belarus'                                      => 'BLR',
		'Belgium'                                      => 'BEL',
		'Belize'                                       => 'BLZ',
		'Benin'                                        => 'BEN',
		'Bermuda'                                      => 'BMU',
		'Bhutan'                                       => 'BTN',
		'Bolivia'                                      => 'BOL',
		'Bonaire, Sint Eustatius and Saba'             => 'BES',
		'Bosnia and Herzegovina'                       => 'BIH',
		'Botswana'                                     => 'BWA',
		'Bouvet Island'                                => 'BVT',
		'Brazil'                                       => 'BRA',
		'Brasil'                                       => 'BRA',
		'British Indian Ocean Territory'               => 'IOT',
		'Brunei Darussalam'                            => 'BRN',
		'Bulgaria'                                     => 'BGR',
		'Burkina Faso'                                 => 'BFA',
		'Burundi'                                      => 'BDI',
		'Cabo Verde'                                   => 'CPV',
		'Cambodia'                                     => 'KHM',
		'Cameroon'                                     => 'CMR',
		'Canada'                                       => 'CAN',
		'Cayman Islands'                               => 'CYM',
		'Central African Republic'                     => 'CAF',
		'Chad'                                         => 'TCD',
		'Chile'                                        => 'CHL',
		'China'                                        => 'CHN',
		'Christmas Island'                             => 'CXR',
		'Cocos (Keeling) Islands'                      => 'CCK',
		'Colombia'                                     => 'COL',
		'Comoros'                                      => 'COM',
		'Congo'                                        => 'COG',
		'Congo, Democratic Republic of the'            => 'COD',
		'Cook Islands'                                 => 'COK',
		'Costa Rica'                                   => 'CRI',
		'Croatia'                                      => 'HRV',
		'Cuba'                                         => 'CUB',
		'Curacao'                                      => 'CUW',
		'Cyprus'                                       => 'CYP',
		'Czech Republic'                               => 'CZE',
		'Denmark'                                      => 'DNK',
		'Djibouti'                                     => 'DJI',
		'Dominica'                                     => 'DMA',
		'Dominican Republic'                           => 'DOM',
		'Ecuador'                                      => 'ECU',
		'Egypt'                                        => 'EGY',
		'El Salvador'                                  => 'SLV',
		'Equatorial Guinea'                            => 'GNQ',
		'Eritrea'                                      => 'ERI',
		'Estonia'                                      => 'EST',
		'Eswatini'                                     => 'SWZ',
		'Ethiopia'                                     => 'ETH',
		'Falkland Islands (Malvinas)'                  => 'FLK',
		'Faroe Islands'                                => 'FRO',
		'Fiji'                                         => 'FJI',
		'Finland'                                      => 'FIN',
		'France'                                       => 'FRA',
		'French Guiana'                                => 'GUF',
		'French Polynesia'                             => 'PYF',
		'French Southern Territories'                  => 'ATF',
		'Gabon'                                        => 'GAB',
		'Gambia'                                       => 'GMB',
		'Georgia'                                      => 'GEO',
		'Germany'                                      => 'DEU',
		'Ghana'                                        => 'GHA',
		'Gibraltar'                                    => 'GIB',
		'Greece'                                       => 'GRC',
		'Greenland'                                    => 'GRL',
		'Grenada'                                      => 'GRD',
		'Guadeloupe'                                   => 'GLP',
		'Guam'                                         => 'GUM',
		'Guatemala'                                    => 'GTM',
		'Guernsey'                                     => 'GGY',
		'Guinea'                                       => 'GIN',
		'Guinea-Bissau'                                => 'GNB',
		'Guyana'                                       => 'GUY',
		'Haiti'                                        => 'HTI',
		'Heard Island and McDonald Islands'            => 'HMD',
		'Holy See'                                     => 'VAT',
		'Honduras'                                     => 'HND',
		'Hong Kong'                                    => 'HKG',
		'Hungary'                                      => 'HUN',
		'Iceland'                                      => 'ISL',
		'India'                                        => 'IND',
		'Indonesia'                                    => 'IDN',
		'Iran, Islamic Republic of'                    => 'IRN',
		'Iraq'                                         => 'IRQ',
		'Ireland'                                      => 'IRL',
		'Isle of Man'                                  => 'IMN',
		'Israel'                                       => 'ISR',
		'Italy'                                        => 'ITA',
		'Jamaica'                                      => 'JAM',
		'Japan'                                        => 'JPN',
		'Jersey'                                       => 'JEY',
		'Jordan'                                       => 'JOR',
		'Kazakhstan'                                   => 'KAZ',
		'Kenya'                                        => 'KEN',
		'Kiribati'                                     => 'KIR',
		'Korea, Democratic People\'s Republic of'      => 'PRK',
		'Korea, Republic of'                           => 'KOR',
		'Kuwait'                                       => 'KWT',
		'Kyrgyzstan'                                   => 'KGZ',
		'Lao People\'s Democratic Republic'            => 'LAO',
		'Latvia'                                       => 'LVA',
		'Lebanon'                                      => 'LBN',
		'Lesotho'                                      => 'LSO',
		'Liberia'                                      => 'LBR',
		'Libya'                                        => 'LBY',
		'Liechtenstein'                                => 'LIE',
		'Lithuania'                                    => 'LTU',
		'Luxembourg'                                   => 'LUX',
		'Macao'                                        => 'MAC',
		'Madagascar'                                   => 'MDG',
		'Malawi'                                       => 'MWI',
		'Malaysia'                                     => 'MYS',
		'Maldives'                                     => 'MDV',
		'Mali'                                         => 'MLI',
		'Malta'                                        => 'MLT',
		'Marshall Islands'                             => 'MHL',
		'Martinique'                                   => 'MTQ',
		'Mauritania'                                   => 'MRT',
		'Mauritius'                                    => 'MUS',
		'Mayotte'                                      => 'MYT',
		'Mexico'                                       => 'MEX',
		'Micronesia, Federated States of'              => 'FSM',
		'Moldova, Republic of'                         => 'MDA',
		'Monaco'                                       => 'MCO',
		'Mongolia'                                     => 'MNG',
		'Montenegro'                                   => 'MNE',
		'Montserrat'                                   => 'MSR',
		'Morocco'                                      => 'MAR',
		'Mozambique'                                   => 'MOZ',
		'Myanmar'                                      => 'MMR',
		'Namibia'                                      => 'NAM',
		'Nauru'                                        => 'NRU',
		'Nepal'                                        => 'NPL',
		'Netherlands'                                  => 'NLD',
		'New Caledonia'                                => 'NCL',
		'New Zealand'                                  => 'NZL',
		'Nicaragua'                                    => 'NIC',
		'Niger'                                        => 'NER',
		'Nigeria'                                      => 'NGA',
		'Niue'                                         => 'NIU',
		'Norfolk Island'                               => 'NFK',
		'North Macedonia'                              => 'MKD',
		'Northern Mariana Islands'                     => 'MNP',
		'Norway'                                       => 'NOR',
		'Oman'                                         => 'OMN',
		'Pakistan'                                     => 'PAK',
		'Palau'                                        => 'PLW',
		'Palestine, State of'                          => 'PSE',
		'Panama'                                       => 'PAN',
		'Papua New Guinea'                             => 'PNG',
		'Paraguay'                                     => 'PRY',
		'Peru'                                         => 'PER',
		'Philippines'                                  => 'PHL',
		'Pitcairn'                                     => 'PCN',
		'Poland'                                       => 'POL',
		'Portugal'                                     => 'PRT',
		'Puerto Rico'                                  => 'PRI',
		'Qatar'                                        => 'QAT',
		'Reunion'                                      => 'REU',
		'Romania'                                      => 'ROU',
		'Russian Federation'                           => 'RUS',
		'Rwanda'                                       => 'RWA',
		'Saint Barthelemy'                             => 'BLM',
		'Saint Helena, Ascension and Tristan da Cunha' => 'SHN',
		'Saint Kitts and Nevis'                        => 'KNA',
		'Saint Lucia'                                  => 'LCA',
		'Saint Martin (French part)'                   => 'MAF',
		'Saint Pierre and Miquelon'                    => 'SPM',
		'Saint Vincent and the Grenadines'             => 'VCT',
		'Samoa'                                        => 'WSM',
		'San Marino'                                   => 'SMR',
		'Sao Tome and Principe'                        => 'STP',
		'Saudi Arabia'                                 => 'SAU',
		'Senegal'                                      => 'SEN',
		'Serbia'                                       => 'SRB',
		'Seychelles'                                   => 'SYC',
		'Sierra Leone'                                 => 'SLE',
		'Singapore'                                    => 'SGP',
		'Sint Maarten (Dutch part)'                    => 'SXM',
		'Slovakia'                                     => 'SVK',
		'Slovenia'                                     => 'SVN',
		'Solomon Islands'                              => 'SLB',
		'Somalia'                                      => 'SOM',
		'South Africa'                                 => 'ZAF',
		'South Georgia and the South Sandwich Islands' => 'SGS',
		'South Sudan'                                  => 'SSD',
		'Spain'                                        => 'ESP',
		'Sri Lanka'                                    => 'LKA',
		'Sudan'                                        => 'SDN',
		'Suriname'                                     => 'SUR',
		'Svalbard and Jan Mayen'                       => 'SJM',
		'Sweden'                                       => 'SWE',
		'Switzerland'                                  => 'CHE',
		'Syrian Arab Republic'                         => 'SYR',
		'Taiwan, Province of China'                    => 'TWN',
		'Tajikistan'                                   => 'TJK',
		'Tanzania, United Republic of'                 => 'TZA',
		'Thailand'                                     => 'THA',
		'Timor-Leste'                                  => 'TLS',
		'Togo'                                         => 'TGO',
		'Tokelau'                                      => 'TKL',
		'Tonga'                                        => 'TON',
		'Trinidad and Tobago'                          => 'TTO',
		'Tunisia'                                      => 'TUN',
		'Turkey'                                       => 'TUR',
		'Türkiye'                                      => 'TUR',
		'Turkmenistan'                                 => 'TKM',
		'Turks and Caicos Islands'                     => 'TCA',
		'Tuvalu'                                       => 'TUV',
		'Uganda'                                       => 'UGA',
		'Ukraine'                                      => 'UKR',
		'United Arab Emirates'                         => 'ARE',
		'United Kingdom of Great Britain and Northern Ireland' => 'GBR',
		'United Kingdom'                               => 'GBR',
		'United States of America'                     => 'USA',
		'United States'                                => 'USA',
		'Uruguay'                                      => 'URY',
		'Uzbekistan'                                   => 'UZB',
		'Vanuatu'                                      => 'VUT',
		'Venezuela (Bolivarian Republic of)'           => 'VEN',
		'Viet Nam'                                     => 'VNM',
		'Virgin Islands (British)'                     => 'VGB',
		'Virgin Islands (U.S.)'                        => 'VIR',
		'Wallis and Futuna'                            => 'WLF',
		'Western Sahara'                               => 'ESH',
		'Yemen'                                        => 'YEM',
		'Zambia'                                       => 'ZMB',
		'Zimbabwe'                                     => 'ZWE',
	);

	/**
	 * Mapping of ISO 3166-1 alpha-3 codes to alpha-2 codes.
	 *
	 * @var array
	 */
	private $alpha3_to_alpha2 = array(
		'AFG' => 'AF',
		'ALA' => 'AX',
		'ALB' => 'AL',
		'DZA' => 'DZ',
		'ASM' => 'AS',
		'AND' => 'AD',
		'AGO' => 'AO',
		'AIA' => 'AI',
		'ATA' => 'AQ',
		'ATG' => 'AG',
		'ARG' => 'AR',
		'ARM' => 'AM',
		'ABW' => 'AW',
		'AUS' => 'AU',
		'AUT' => 'AT',
		'AZE' => 'AZ',
		'BHS' => 'BS',
		'BHR' => 'BH',
		'BGD' => 'BD',
		'BRB' => 'BB',
		'BLR' => 'BY',
		'BEL' => 'BE',
		'BLZ' => 'BZ',
		'BEN' => 'BJ',
		'BMU' => 'BM',
		'BTN' => 'BT',
		'BOL' => 'BO',
		'BES' => 'BQ',
		'BIH' => 'BA',
		'BWA' => 'BW',
		'BVT' => 'BV',
		'BRA' => 'BR',
		'IOT' => 'IO',
		'BRN' => 'BN',
		'BGR' => 'BG',
		'BFA' => 'BF',
		'BDI' => 'BI',
		'CPV' => 'CV',
		'KHM' => 'KH',
		'CMR' => 'CM',
		'CAN' => 'CA',
		'CYM' => 'KY',
		'CAF' => 'CF',
		'TCD' => 'TD',
		'CHL' => 'CL',
		'CHN' => 'CN',
		'CXR' => 'CX',
		'CCK' => 'CC',
		'COL' => 'CO',
		'COM' => 'KM',
		'COG' => 'CG',
		'COD' => 'CD',
		'COK' => 'CK',
		'CRI' => 'CR',
		'HRV' => 'HR',
		'CUB' => 'CU',
		'CUW' => 'CW',
		'CYP' => 'CY',
		'CZE' => 'CZ',
		'DNK' => 'DK',
		'DJI' => 'DJ',
		'DMA' => 'DM',
		'DOM' => 'DO',
		'ECU' => 'EC',
		'EGY' => 'EG',
		'SLV' => 'SV',
		'GNQ' => 'GQ',
		'ERI' => 'ER',
		'EST' => 'EE',
		'SWZ' => 'SZ',
		'ETH' => 'ET',
		'FLK' => 'FK',
		'FRO' => 'FO',
		'FJI' => 'FJ',
		'FIN' => 'FI',
		'FRA' => 'FR',
		'GUF' => 'GF',
		'PYF' => 'PF',
		'ATF' => 'TF',
		'GAB' => 'GA',
		'GMB' => 'GM',
		'GEO' => 'GE',
		'DEU' => 'DE',
		'GHA' => 'GH',
		'GIB' => 'GI',
		'GRC' => 'GR',
		'GRL' => 'GL',
		'GRD' => 'GD',
		'GLP' => 'GP',
		'GUM' => 'GU',
		'GTM' => 'GT',
		'GGY' => 'GG',
		'GIN' => 'GN',
		'GNB' => 'GW',
		'GUY' => 'GY',
		'HTI' => 'HT',
		'HMD' => 'HM',
		'VAT' => 'VA',
		'HND' => 'HN',
		'HKG' => 'HK',
		'HUN' => 'HU',
		'ISL' => 'IS',
		'IND' => 'IN',
		'IDN' => 'ID',
		'IRN' => 'IR',
		'IRQ' => 'IQ',
		'IRL' => 'IE',
		'IMN' => 'IM',
		'ISR' => 'IL',
		'ITA' => 'IT',
		'JAM' => 'JM',
		'JPN' => 'JP',
		'JEY' => 'JE',
		'JOR' => 'JO',
		'KAZ' => 'KZ',
		'KEN' => 'KE',
		'KIR' => 'KI',
		'PRK' => 'KP',
		'KOR' => 'KR',
		'KWT' => 'KW',
		'KGZ' => 'KG',
		'LAO' => 'LA',
		'LVA' => 'LV',
		'LBN' => 'LB',
		'LSO' => 'LS',
		'LBR' => 'LR',
		'LBY' => 'LY',
		'LIE' => 'LI',
		'LTU' => 'LT',
		'LUX' => 'LU',
		'MAC' => 'MO',
		'MDG' => 'MG',
		'MWI' => 'MW',
		'MYS' => 'MY',
		'MDV' => 'MV',
		'MLI' => 'ML',
		'MLT' => 'MT',
		'MHL' => 'MH',
		'MTQ' => 'MQ',
		'MRT' => 'MR',
		'MUS' => 'MU',
		'MYT' => 'YT',
		'MEX' => 'MX',
		'FSM' => 'FM',
		'MDA' => 'MD',
		'MCO' => 'MC',
		'MNG' => 'MN',
		'MNE' => 'ME',
		'MSR' => 'MS',
		'MAR' => 'MA',
		'MOZ' => 'MZ',
		'MMR' => 'MM',
		'NAM' => 'NA',
		'NRU' => 'NR',
		'NPL' => 'NP',
		'NLD' => 'NL',
		'NCL' => 'NC',
		'NZL' => 'NZ',
		'NIC' => 'NI',
		'NER' => 'NE',
		'NGA' => 'NG',
		'NIU' => 'NU',
		'NFK' => 'NF',
		'MKD' => 'MK',
		'MNP' => 'MP',
		'NOR' => 'NO',
		'OMN' => 'OM',
		'PAK' => 'PK',
		'PLW' => 'PW',
		'PSE' => 'PS',
		'PAN' => 'PA',
		'PNG' => 'PG',
		'PRY' => 'PY',
		'PER' => 'PE',
		'PHL' => 'PH',
		'PCN' => 'PN',
		'POL' => 'PL',
		'PRT' => 'PT',
		'PRI' => 'PR',
		'QAT' => 'QA',
		'REU' => 'RE',
		'ROU' => 'RO',
		'RUS' => 'RU',
		'RWA' => 'RW',
		'BLM' => 'BL',
		'SHN' => 'SH',
		'KNA' => 'KN',
		'LCA' => 'LC',
		'MAF' => 'MF',
		'SPM' => 'PM',
		'VCT' => 'VC',
		'WSM' => 'WS',
		'SMR' => 'SM',
		'STP' => 'ST',
		'SAU' => 'SA',
		'SEN' => 'SN',
		'SRB' => 'RS',
		'SYC' => 'SC',
		'SLE' => 'SL',
		'SGP' => 'SG',
		'SXM' => 'SX',
		'SVK' => 'SK',
		'SVN' => 'SI',
		'SLB' => 'SB',
		'SOM' => 'SO',
		'ZAF' => 'ZA',
		'SGS' => 'GS',
		'SSD' => 'SS',
		'ESP' => 'ES',
		'LKA' => 'LK',
		'SDN' => 'SD',
		'SUR' => 'SR',
		'SJM' => 'SJ',
		'SWE' => 'SE',
		'CHE' => 'CH',
		'SYR' => 'SY',
		'TWN' => 'TW',
		'TJK' => 'TJ',
		'TZA' => 'TZ',
		'THA' => 'TH',
		'TLS' => 'TL',
		'TGO' => 'TG',
		'TKL' => 'TK',
		'TON' => 'TO',
		'TTO' => 'TT',
		'TUN' => 'TN',
		'TUR' => 'TR',
		'TKM' => 'TM',
		'TCA' => 'TC',
		'TUV' => 'TV',
		'UGA' => 'UG',
		'UKR' => 'UA',
		'ARE' => 'AE',
		'GBR' => 'GB',
		'USA' => 'US',
		'URY' => 'UY',
		'UZB' => 'UZ',
		'VUT' => 'VU',
		'VEN' => 'VE',
		'VNM' => 'VN',
		'VGB' => 'VG',
		'VIR' => 'VI',
		'WLF' => 'WF',
		'ESH' => 'EH',
		'YEM' => 'YE',
		'ZMB' => 'ZM',
		'ZWE' => 'ZW',
	);

	/**
	 * Mapping of US state names to ISO 3166-2 abbreviations.
	 *
	 * @var array
	 */
	private $state_names = array(
		'Alabama'                      => 'AL',
		'Alaska'                       => 'AK',
		'Arizona'                      => 'AZ',
		'Arkansas'                     => 'AR',
		'California'                   => 'CA',
		'Colorado'                     => 'CO',
		'Connecticut'                  => 'CT',
		'Delaware'                     => 'DE',
		'Florida'                      => 'FL',
		'Georgia'                      => 'GA',
		'Hawaii'                       => 'HI',
		'Idaho'                        => 'ID',
		'Illinois'                     => 'IL',
		'Indiana'                      => 'IN',
		'Iowa'                         => 'IA',
		'Kansas'                       => 'KS',
		'Kentucky'                     => 'KY',
		'Louisiana'                    => 'LA',
		'Maine'                        => 'ME',
		'Maryland'                     => 'MD',
		'Massachusetts'                => 'MA',
		'Michigan'                     => 'MI',
		'Minnesota'                    => 'MN',
		'Mississippi'                  => 'MS',
		'Missouri'                     => 'MO',
		'Montana'                      => 'MT',
		'Nebraska'                     => 'NE',
		'Nevada'                       => 'NV',
		'New Hampshire'                => 'NH',
		'New Jersey'                   => 'NJ',
		'New Mexico'                   => 'NM',
		'New York'                     => 'NY',
		'North Carolina'               => 'NC',
		'North Dakota'                 => 'ND',
		'Ohio'                         => 'OH',
		'Oklahoma'                     => 'OK',
		'Oregon'                       => 'OR',
		'Pennsylvania'                 => 'PA',
		'Rhode Island'                 => 'RI',
		'South Carolina'               => 'SC',
		'South Dakota'                 => 'SD',
		'Tennessee'                    => 'TN',
		'Texas'                        => 'TX',
		'Utah'                         => 'UT',
		'Vermont'                      => 'VT',
		'Virginia'                     => 'VA',
		'Washington'                   => 'WA',
		'West Virginia'                => 'WV',
		'Wisconsin'                    => 'WI',
		'Wyoming'                      => 'WY',
		'District of Columbia'         => 'DC',
		'Puerto Rico'                  => 'PR',
		'Guam'                         => 'GU',
		'American Samoa'               => 'AS',
		'Northern Mariana Islands'     => 'MP',
		'United States Virgin Islands' => 'VI',
	);

	/**
	 * Converts country names to ISO 3166-1 alpha-3 codes.
	 *
	 * @since 3.44.3
	 *
	 * @param string $name Country name or abbreviation.
	 * @return string|false ISO 3166-1 alpha-3 or alpha-2 code or false if not found.
	 */
	public function country_to_alpha( $name, $format = 'alpha-3' ) {

		$value = false;

		if ( ( 3 === strlen( $name ) && 'alpha-3' === $format ) || ( 2 === strlen( $name ) && 'alpha-2' === $format ) ) {

			// it's already in the correct format.

			$value = strtoupper( $name );

		} elseif ( 2 === strlen( $name ) && 'alpha-3' === $format ) {

			// Convert alpha-2 to alpha-3.

			$value = array_search( strtoupper( $name ), $this->alpha3_to_alpha2, true );

		} elseif ( 3 === strlen( $name ) && 'alpha-2' === $format ) {

			// Convert alpha-3 to alpha-2.

			if ( isset( $this->alpha3_to_alpha2[ $name ] ) ) {
				$value = $this->alpha3_to_alpha2[ $name ];
			}
		} else {

			$country_names = array_change_key_case( $this->name_to_alpha3, CASE_LOWER );
			$value_lower   = strtolower( $name );

			if ( isset( $country_names[ $value_lower ] ) ) {
				$value = $country_names[ $value_lower ];

				if ( 'alpha-2' === $format ) {
					$value = $this->alpha3_to_alpha2[ $value ];
				}
			}
		}

		return apply_filters( 'wpf_country_to_iso3166', $value, $name, $format );
	}

	/**
	 * Converts state names to their ISO 3166-2 code.
	 *
	 * @since 3.44.3
	 *
	 * @param string $name State name.
	 * @return string|false ISO 3166-2 code or false if not found.
	 */
	public function state_name_to_alpha( $name ) {

		if ( 2 === strlen( $name ) ) {
			return strtoupper( $name ); // it's already an ISO 3166-2 code.
		}

		$state_names = array_change_key_case( $this->state_names, CASE_LOWER );
		$value_lower = strtolower( $name );

		if ( isset( $state_names[ $value_lower ] ) ) {
			return $state_names[ $value_lower ];
		}
		return false; // Return false if the state name is not found
	}

	/**
	 * Checks if a state code is a US state.
	 *
	 * @since 3.44.5
	 *
	 * @param string $code The state code.
	 * @return bool Whether the state code is a US state.
	 */
	public function is_us_state( $code ) {
		return in_array( strtoupper( $code ), array_values( $this->state_names ), true );
	}
}
