import { useRef } from 'react';
import { useSelector, useDispatch } from 'react-redux';
import { useLocation } from 'react-router';
import { changeState } from '../../store/actions';
import { usePostData, contactNewsletterData } from '../../helpers/hooks';
import { useInView } from 'react-intersection-observer';
import { useMediaQuery } from '../../helpers/hooks';
import { Container, ScrollTopBtn, PreloaderNewsletter, Logo } from '../../components';
import { Facebook, Linkedin, Instagram, Youtube, Arrow } from '../../assets';
import './style.scss';
import Swal from 'sweetalert2';

const Footer = () => {
  const breakpointLg = useMediaQuery('lg');
  const { pathname } = useLocation();
  const { ref, inView } = useInView({
    threshold: 0.25,
    triggerOnce: true,
  });
  const dispatch = useDispatch();
  const { showNewsletterEmailRequired, showNewsletterInvalidEmail, webpSupport, isIntroEnded, isMainContentLoaded } =
    useSelector((state) => state.reducers);
  const form = useRef();
  const socialNetworks = [
    {
      img: Facebook,
      link: 'https://www.facebook.com/EpicaCreative',
      alt: 'Facebook logotype',
    },
    {
      img: Linkedin,
      link: 'https://www.linkedin.com/company/uosolutions/',
      alt: 'Linkedin logotype',
    },
    {
      img: Instagram,
      link: 'https://www.instagram.com/epica.creative/',
      alt: 'Instagram logotype',
    },
    {
      img: Youtube,
      link: 'https://www.youtube.com/channel/UCm59nRgnNkzwE_77L5jJ8FA',
      alt: 'YouTube logotype',
    },
  ];
  const onSuccess = () => {
    Swal.fire({
      icon: 'success',
      title: 'Thanks for subscribing!',
      text: "You're now subscribed to our newsletter!. You will hear from us shortly.",
      showConfirmButton: true,
      confirmButtonText: 'Close',
    });
    form.current.email.value = '';
  };
  const onError = () => {
    Swal.fire({
      icon: 'error',
      title: 'Error!',
      text: 'There was an error trying to send your message. Please try again later.',
      showConfirmButton: true,
      confirmButtonText: 'Close',
    });
  };
  const { mutate, isLoading } = usePostData(contactNewsletterData, onSuccess, onError);

  const sendEmail = (e) => {
    e.preventDefault();
    const validated = validateInputs();
    const dataLayerInfo = {
      categoria: 'epica',
      tipo: 'newsletter',
      status: validated ? 'Completado' : 'No Completado',
    };
    !validated && (dataLayerInfo.mensaje = 'Faltan campos por completar: email');

    window.dataLayer = window.dataLayer || [];
    window.dataLayer.push({
      event: 'ev_form',
      data: dataLayerInfo,
    });

    if (!validated) return;
    mutate({ email: form.current.email.value });
  };

  const validateInputs = () => {
    const emailRegex = /^[-\w.%+]{1,64}@(?:[A-Z0-9-]{1,63}\.){1,125}[A-Z]{2,63}$/i;
    if (form.current.email.value === '') {
      dispatch(changeState('showNewsletterEmailRequired', true));
      return false;
    }
    if (!emailRegex.test(form.current.email.value) && form.current.email.value !== '') {
      dispatch(changeState('showNewsletterInvalidEmail', true));
      return false;
    }
    return true;
  };

  return (
    <>
      {(isIntroEnded || isMainContentLoaded) && (
        <footer className='footer' ref={ref}>
          {breakpointLg && <ScrollTopBtn />}
          <Container>
            <div className='epica-logo-container'>
              <div className={`img-text-container${inView ? ' active' : ''}`}>
                <div className='epica-logo'>
                  <Logo />
                </div>
                <p>
                  a{' '}
                  <a href='https://uosolutions.com/' target='_blank' rel='noopener noreferrer'>
                    UO SOLUTIONS GROUP
                  </a>{' '}
                  Company.
                </p>
              </div>
            </div>
            <div className={`newsletter-container${inView ? ' active' : ''}`}>
              <div className='newsletter-copyright'>
                <form ref={form} onSubmit={sendEmail}>
                  <div>
                    Newsletter
                    <div className='newsletter-input-group'>
                      <span className={`newsletter-input-box${isLoading ? ' newsletter-input-box--disabled' : ''}`}>
                        <input
                          type='text'
                          placeholder='Your email goes here :)'
                          name='email'
                          onChange={() => {
                            dispatch(changeState('showNewsletterEmailRequired', false));
                            dispatch(changeState('showNewsletterInvalidEmail', false));
                          }}
                        />
                        <span
                          style={{
                            display: showNewsletterInvalidEmail ? 'block' : 'none',
                          }}
                        >
                          Invalid Email address.
                        </span>
                        <span
                          style={{
                            display: showNewsletterEmailRequired ? 'block' : 'none',
                          }}
                        >
                          This field is required.
                        </span>
                      </span>
                      {isLoading ? (
                        <PreloaderNewsletter className='newsletter-loader' />
                      ) : (
                        <button type='submit'>
                          <img src={Arrow} alt='Click the white arrow to submit the form' />
                        </button>
                      )}
                    </div>
                  </div>
                </form>
                <p className='newsletter-copyright__paragraph'>
                  <span className='newsletter-copyright__paragraph-span'>
                    Copyright © {new Date().getFullYear()} Epica Creative all rights reserved
                  </span>
                  {pathname !== '/privacy-policy' && (
                    <a href='/privacy-policy' target='_blank' rel='noopener noreferrer' className='toPrivacyPolicy'>
                      Privacy policy
                    </a>
                  )}
                </p>
              </div>
              <div className='social-info'>
                <p>
                  <a
                    href='mailto:info@epicacreative.com?subject=Contact from epicacreative.com'
                    className='social-info__mailto'
                  >
                    info@epicacreative.com
                  </a>
                </p>
                <div className='social-info__social-networks'>
                  {socialNetworks.map((social, i) => (
                    <a
                      href={social.link}
                      key={i}
                      className='social-info__link'
                      target='_blank'
                      rel='noopener noreferrer'
                    >
                      <img src={social.img} alt={social.alt} className='social-info__img' />
                    </a>
                  ))}
                </div>
                <p className='newsletter-copyright__paragraph-bottom'>
                  Copyright © {new Date().getFullYear()} Epica Creative all rights reserved
                </p>
              </div>
            </div>
          </Container>
        </footer>
      )}
    </>
  );
};

export default Footer;
