import {
  DataTexture
} from 'three'

/**
 * @class ShaderFloatArray
 *
 * When writing a custom WebGL shader, sometimes you need to pass it an array of floating
 * point numbers that it can read from. Unfortunately this is very difficult to do in WebGL,
 * because:
 *
 *   - GLSL "array" uniforms can only be of a constant length.
 *   - Textures can only hold floating point numbers in WebGL1 if the `OES_texture_float`
 *     extension is available.
 *
 * ShaderFloatArray is an array-like abstraction that encodes its floating point data into
 * an RGBA texture's four Uint8 components, and provides the corresponding ThreeJS uniforms
 * and GLSL code for you to put in your custom shader to query the float values by array index.
 *
 * This should generally only be used within a fragment shader, as some environments (e.g. iOS)
 * only allow texture lookups in fragment shaders.
 *
 * TODO:
 *   - Fix texture to fill both dimensions so we don't easily hit max texture size limits
 *   - Use a float texture if the extension is available so we can skip the encoding process
 */
export class ShaderFloatArray {
  constructor(name) {
    this.name = name
    this.textureUniform = `dataTex_${name}`
    this.textureSizeUniform = `dataTexSize_${name}`
    this.multiplierUniform = `dataMultiplier_${name}`

    /**
     * @property dataSizeUniform - the name of the GLSL uniform that will hold the
     * length of the data array.
     * @type {string}
     */
    this.dataSizeUniform = `dataSize_${name}`

    /**
     * @property readFunction - the name of the GLSL function that should be called to
     * read data out of the array by index.
     * @type {string}
     */
    this.readFunction = `readData_${name}`

    this._raw = new Float32Array(0)
    this._texture = new DataTexture(new Uint8Array(0), 0, 1)
    this._length = 0
    this._multiplier = 1
  }

  /**
   * @property length - the current length of the data array
   * @type {number}
   */
  set length(value) {
    if (value !== this._length) {
      // Find nearest power-of-2 that holds the new length
      const size = Math.pow(2, Math.ceil(Math.log2(value)))
      const raw = this._raw
      if (size < raw.length) {
        this._raw = raw.subarray(0, size)
      }
      else if(size > raw.length) {
        this._raw = new Float32Array(size)
        this._raw.set(raw)
      }
      this._length = value
    }
  }
  get length() {
    return this._length
  }

  /**
   * Add a value to the end of the data array
   * @param {number} value
   */
  push(value) {
    return this.set(this.length++, value)
  }

  /**
   * Replace the existing data with that from a new array
   * @param {ArrayLike<number>} array
   */
  setArray(array) {
    this.length = array.length
    this._raw.set(array)
    this._needsRepack = true
  }

  /**
   * Get the current value at index
   * @param {number} index
   * @return {number}
   */
  get(index) {
    return this._raw[index]
  }

  set(index, value) {
    if (index + 1 > this._length) {
      this.length = index + 1
    }
    if (value !== this._raw[index]) {
      this._raw[index] = value
      encodeFloatToFourInts(
        value / this._multiplier,
        this._texture.image.data,
        index * 4
      )
      this._needsMultCheck = true
    }
  }

  /**
   * Make a copy of this ShaderFloatArray
   * @return {ShaderFloatArray}
   */
  clone() {
    const clone = new ShaderFloatArray(this.name)
    clone.setArray(this._raw)
    return clone
  }

  /**
   * Retrieve the set of Uniforms that must to be added to the target ShaderMaterial or
   * DerivedMaterial, to feed the GLSL code generated by {@link #getShaderHeaderCode}.
   * @return {Object}
   */
  getShaderUniforms() {
    const me = this
    return {
      [this.textureUniform]: {get value() {
        me._sync()
        return me._texture
      }},
      [this.textureSizeUniform]: {get value() {
        me._sync()
        return me._texture.image.width
      }},
      [this.dataSizeUniform]: {get value() {
        me._sync()
        return me.length
      }},
      [this.multiplierUniform]: {get value() {
        me._sync()
        return me._multiplier
      }}
    }
  }

  /**
   * Retrieve the GLSL code that must be injected into the shader's definitions area to
   * enable reading from the data array. This exposes a function with a name matching
   * the {@link #readFunction} property, which other shader code can call to read values
   * from the array by their index.
   * @return {string}
   */
  getShaderHeaderCode() {
    const {textureUniform, textureSizeUniform, dataSizeUniform, multiplierUniform, readFunction} = this
    return `
uniform sampler2D ${textureUniform};
uniform float ${textureSizeUniform};
uniform float ${dataSizeUniform};
uniform float ${multiplierUniform};

float ${readFunction}(float index) {
  vec2 texUV = vec2((index + 0.5) / ${textureSizeUniform}, 0.5);
  vec4 pixel = texture2D(${textureUniform}, texUV);
  return dot(pixel, 1.0 / vec4(1.0, 255.0, 65025.0, 16581375.0)) * ${multiplierUniform};
}
`
  }

  /**
   * @private Synchronize any pending changes to the underlying DataTexture
   */
  _sync() {
    const tex = this._texture
    const raw = this._raw
    let needsRepack = this._needsRepack

    // If the size of the raw array changed, resize the texture to match
    if (raw.length !== tex.image.width) {
      tex.image = {
        data: new Uint8Array(raw.length * 4),
        width: raw.length,
        height: 1
      }
      needsRepack = true
    }

    // If the values changed, check the multiplier. This should be a value by which
    // all the values are divided to constrain them to the [0,1] range required by
    // the Uint8 packing algorithm. We pick the nearest power of 2 that holds the
    // maximum value for greatest accuracy.
    if (needsRepack || this._needsMultCheck) {
      const maxVal = this._raw.reduce((a, b) => Math.max(a, b), 0)
      const mult = Math.pow(2, Math.ceil(Math.log2(maxVal)))
      if (mult !== this._multiplier) {
        this._multiplier = mult
        needsRepack = true
      }
      tex.needsUpdate = true
      this._needsMultCheck = false
    }

    // If things changed in a way we need to repack, do so
    if (needsRepack) {
      for (let i = 0, len = raw.length, mult = this._multiplier; i < len; i++) {
        encodeFloatToFourInts(raw[i] / mult, tex.image.data, i * 4)
      }
      this._needsRepack = false
    }
  }
}



/**
 * Encode a floating point number into a set of four 8-bit integers.
 * Also see the companion decoder function #decodeFloatFromFourInts.
 *
 * This is adapted to JavaScript from the basic approach at
 * http://aras-p.info/blog/2009/07/30/encoding-floats-to-rgba-the-final/
 * but writes out integers in the range 0-255 instead of floats in the range 0-1
 * so they can be more easily used in a Uint8Array for standard WebGL rgba textures.
 *
 * Some precision will necessarily be lost during the encoding and decoding process.
 * Testing shows that the maximum precision error is ~1.18e-10 which should be good
 * enough for most cases.
 *
 * @param {Number} value - the floating point number to encode. Must be in the range [0, 1]
 *        otherwise the results will be incorrect.
 * @param {Array|Uint8Array} array - an array into which the four ints should be written
 * @param {Number} startIndex - index in the output array at which to start writing the ints
 * @return {Array|Uint8Array}
 */
export function encodeFloatToFourInts(value, array, startIndex) {
  // This is adapted to JS from the basic approach at
  // http://aras-p.info/blog/2009/07/30/encoding-floats-to-rgba-the-final/
  // but writes to a Uint8Array instead of floats. Input values must be in
  // the range [0, 1]. The maximum error after encoding and decoding is ~1.18e-10
  let enc0 = 255 * value
  let enc1 = 255 * (enc0 % 1)
  let enc2 = 255 * (enc1 % 1)
  let enc3 = 255 * (enc2 % 1)

  enc0 = enc0 & 255
  enc1 = enc1 & 255
  enc2 = enc2 & 255
  enc3 = Math.round(enc3) & 255

  array[startIndex] = enc0
  array[startIndex + 1] = enc1
  array[startIndex + 2] = enc2
  array[startIndex + 3] = enc3
  return array
}

/**
 * Companion to #encodeFloatToFourInts
 * @param {Array|Uint8Array} array - an array holding the ints to read from
 * @param {Number} startIndex - index in the array at which to start reading
 */
export function decodeFloatFromFourInts(array, startIndex) {
  return array[startIndex] / 255
    + array[startIndex + 1] / 65025 //255**2
    + array[startIndex + 2] / 16581375 //255**3
    + array[startIndex + 3] / 4228250625 //255**4
}

