import _extends from '@babel/runtime/helpers/esm/extends';
import * as React from 'react';
import { useThree, useLoader, useFrame, createPortal } from '@react-three/fiber';
import { CubeReflectionMapping, EquirectangularReflectionMapping, sRGBEncoding, LinearEncoding, Scene, WebGLCubeRenderTarget, HalfFloatType, CubeTextureLoader, BackSide } from 'three';
import { RGBELoader } from 'three-stdlib';
import { presetsObj } from '../helpers/environment-assets.js';
import vertexShader from '../helpers/glsl/GroundProjection.vert.glsl.js';
import fragmentShader from '../helpers/glsl/GroundProjection.frag.glsl.js';
import { Icosahedron } from './shapes.js';

const CUBEMAP_ROOT = 'https://market-assets.fra1.cdn.digitaloceanspaces.com/market-assets/hdris/';

const isCubeTexture = def => def && def.isCubeTexture;

const isRef = obj => obj.current && obj.current.isScene;

const resolveScene = scene => isRef(scene) ? scene.current : scene;

function EnvironmentMap({
  scene,
  background = false,
  map
}) {
  const defaultScene = useThree(state => state.scene);
  React.useLayoutEffect(() => {
    if (map) {
      const target = resolveScene(scene || defaultScene);
      const oldbg = target.background;
      const oldenv = target.environment;
      if (background !== 'only') target.environment = map;
      if (background) target.background = map;
      return () => {
        if (background !== 'only') target.environment = oldenv;
        if (background) target.background = oldbg;
      };
    }
  }, [defaultScene, scene, map, background]);
  return null;
}
function useEnvironment({
  files = ['/px.png', '/nx.png', '/py.png', '/ny.png', '/pz.png', '/nz.png'],
  path = '',
  preset = undefined,
  encoding = undefined,
  extensions
}) {
  if (preset) {
    if (!(preset in presetsObj)) throw new Error('Preset must be one of: ' + Object.keys(presetsObj).join(', '));
    files = presetsObj[preset];
    path = CUBEMAP_ROOT;
  }

  const isCubeMap = Array.isArray(files);
  const loader = isCubeMap ? CubeTextureLoader : RGBELoader;
  const loaderResult = useLoader( // @ts-expect-error
  loader, isCubeMap ? [files] : files, loader => {
    loader.setPath(path);
    if (extensions) extensions(loader);
  });
  const texture = isCubeMap ? // @ts-ignore
  loaderResult[0] : loaderResult;
  texture.mapping = isCubeMap ? CubeReflectionMapping : EquirectangularReflectionMapping;
  texture.encoding = (encoding !== null && encoding !== void 0 ? encoding : isCubeMap) ? sRGBEncoding : LinearEncoding;
  return texture;
}
function EnvironmentCube({
  background = false,
  scene,
  ...rest
}) {
  const texture = useEnvironment(rest);
  const defaultScene = useThree(state => state.scene);
  React.useLayoutEffect(() => {
    const target = resolveScene(scene || defaultScene);
    const oldbg = target.background;
    const oldenv = target.environment;
    if (background !== 'only') target.environment = texture;
    if (background) target.background = texture;
    return () => {
      if (background !== 'only') target.environment = oldenv;
      if (background) target.background = oldbg;
    };
  }, [texture, background, scene, defaultScene]);
  return null;
}
function EnvironmentPortal({
  children,
  near = 1,
  far = 1000,
  resolution = 256,
  frames = 1,
  map,
  background = false,
  scene,
  files,
  path,
  preset = undefined,
  extensions
}) {
  const gl = useThree(state => state.gl);
  const defaultScene = useThree(state => state.scene);
  const camera = React.useRef(null);
  const [virtualScene] = React.useState(() => new Scene());
  const fbo = React.useMemo(() => {
    const fbo = new WebGLCubeRenderTarget(resolution);
    fbo.texture.type = HalfFloatType;
    return fbo;
  }, [resolution]);
  React.useLayoutEffect(() => {
    if (frames === 1) camera.current.update(gl, virtualScene);
    const target = resolveScene(scene || defaultScene);
    const oldbg = target.background;
    const oldenv = target.environment;
    if (background !== 'only') target.environment = fbo.texture;
    if (background) target.background = fbo.texture;
    return () => {
      if (background !== 'only') target.environment = oldenv;
      if (background) target.background = oldbg;
    };
  }, [children, virtualScene, fbo.texture, scene, defaultScene, background, frames, gl]);
  let count = 1;
  useFrame(() => {
    if (frames === Infinity || count < frames) {
      camera.current.update(gl, virtualScene);
      count++;
    }
  });
  return /*#__PURE__*/React.createElement(React.Fragment, null, createPortal( /*#__PURE__*/React.createElement(React.Fragment, null, children, /*#__PURE__*/React.createElement("cubeCamera", {
    ref: camera,
    args: [near, far, fbo]
  }), files || preset ? /*#__PURE__*/React.createElement(EnvironmentCube, {
    background: true,
    files: files,
    preset: preset,
    path: path,
    extensions: extensions
  }) : map ? /*#__PURE__*/React.createElement(EnvironmentMap, {
    background: true,
    map: map,
    extensions: extensions
  }) : null), virtualScene));
}

function EnvironmentGround(props) {
  var _props$ground, _props$ground2, _scale, _props$ground3;

  const textureDefault = useEnvironment(props);
  const texture = props.map || textureDefault;
  const isCubeMap = isCubeTexture(texture);
  const defines = React.useMemo(() => {
    var _ref, _texture$image$;

    const w = (_ref = isCubeMap ? (_texture$image$ = texture.image[0]) == null ? void 0 : _texture$image$.width : texture.image.width) !== null && _ref !== void 0 ? _ref : 1024;
    const cubeSize = w / 4;

    const _lodMax = Math.floor(Math.log2(cubeSize));

    const _cubeSize = Math.pow(2, _lodMax);

    const width = 3 * Math.max(_cubeSize, 16 * 7);
    const height = 4 * _cubeSize;
    return [isCubeMap ? `#define ENVMAP_TYPE_CUBE` : '', `#define CUBEUV_TEXEL_WIDTH ${1.0 / width}`, `#define CUBEUV_TEXEL_HEIGHT ${1.0 / height}`, `#define CUBEUV_MAX_MIP ${_lodMax}.0`, ``];
  }, []);
  const fragment = React.useMemo(() => defines.join('\n') + fragmentShader, [defines]);
  const uniforms = React.useMemo(() => ({
    cubemap: {
      value: null
    },
    height: {
      value: 15
    },
    radius: {
      value: 60
    }
  }), []);
  const mat = React.useRef(null);
  const height = (_props$ground = props.ground) == null ? void 0 : _props$ground.height;
  const radius = (_props$ground2 = props.ground) == null ? void 0 : _props$ground2.radius;
  const scale = (_scale = (_props$ground3 = props.ground) == null ? void 0 : _props$ground3.scale) !== null && _scale !== void 0 ? _scale : 1000;
  React.useEffect(() => void (height && (mat.current.uniforms.height.value = height)), [height]);
  React.useEffect(() => void (radius && (mat.current.uniforms.radius.value = radius)), [radius]);
  React.useEffect(() => void (mat.current.uniforms.cubemap.value = texture), [texture]);
  return /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(EnvironmentMap, _extends({}, props, {
    map: texture
  })), /*#__PURE__*/React.createElement(Icosahedron, {
    scale: scale,
    args: [1, 16]
  }, /*#__PURE__*/React.createElement("shaderMaterial", {
    ref: mat,
    side: BackSide,
    vertexShader: vertexShader,
    fragmentShader: fragment,
    uniforms: uniforms
  })));
}

function Environment(props) {
  return props.ground ? /*#__PURE__*/React.createElement(EnvironmentGround, props) : props.map ? /*#__PURE__*/React.createElement(EnvironmentMap, props) : props.children ? /*#__PURE__*/React.createElement(EnvironmentPortal, props) : /*#__PURE__*/React.createElement(EnvironmentCube, props);
}

export { Environment, EnvironmentCube, EnvironmentMap, EnvironmentPortal, useEnvironment };
